/* Code to test real-time latencies on Linux */

/* To use on a Raspberry Pi, connect GPIO24 to GPIO25 via a 1k resistor */

/* This code starts two threads: */
/*      one waits roughly 100ms then sets GPIO24 high while logging the */
/*      time it set it high */
/*	the other monitors GPIO25 and as soon as it goes high it */
/*	logs the time it noticed */


#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <errno.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <string.h>
#include <linux/gpio.h>
#include <limits.h>

#include <pthread.h>

static int max_count=10;

static struct timespec *set_hi;
static struct timespec *got_hi;

/* calculate the difference in time in nanoseconds */
/* between two time measurements */
static long int calc_diff(struct timespec time1, struct timespec time2) {

	long int diff_ns,diff_s;

	diff_ns=time2.tv_nsec-time1.tv_nsec;
	diff_s=time2.tv_sec-time1.tv_sec;
	if (diff_ns<0) {
		diff_s+=1;
		diff_ns+=1000000000;
	}

	return diff_ns;

}

/* initialize the GPIOs */
static int init_gpio(int *fd24, int *fd25) {

	int fd,ioctl_result;
	struct gpiochip_info chip_info;
        struct gpioline_info line_info;
        struct gpiohandle_request req24,req25;

	/* Export the GPIO */
	fd=open("/dev/gpiochip0",O_RDWR);
	if (fd<0) {
		fprintf(stderr,"Error opening %s\n",strerror(errno));
		return -1;
	}

	ioctl_result=ioctl(fd,GPIO_GET_CHIPINFO_IOCTL,&chip_info);
	if (ioctl_result<0 ) {
		printf("Error ioctl %s\n",strerror(errno));
		close(fd);
		return -1;
	}

	printf("Found %s, %s, %d lines\n",
		chip_info.name,chip_info.label,chip_info.lines);


	/* request GPIO24 for output lines */
	memset(&req24,0,sizeof(struct gpiohandle_request));
	req24.flags |= GPIOHANDLE_REQUEST_OUTPUT;
	req24.lines =1;
	req24.lineoffsets[0] =24;
	req24.default_values[0] =0;
	strcpy(req24.consumer_label, "ECE471");
	ioctl_result = ioctl(fd, GPIO_GET_LINEHANDLE_IOCTL, &req24);
	 if (ioctl_result<0 ) {
		fprintf(stderr,"Error ioctl %s\n",strerror(errno));
		close(fd);
		return -1;
	}


	/* request GPIO25 for input */
	memset(&req25,0,sizeof(struct gpiohandle_request));
	req25.flags |= GPIOHANDLE_REQUEST_INPUT;
	req25.lines =1;
	req25.lineoffsets[0] =25;
	req25.default_values[0] =0;
	strcpy(req25.consumer_label, "ECE471");
	ioctl_result = ioctl(fd, GPIO_GET_LINEHANDLE_IOCTL, &req25);
	 if (ioctl_result<0 ) {
		fprintf(stderr,"Error ioctl %s\n",strerror(errno));
		close(fd);
		return -1;
	}


	/* info struct used for lots of things */
	memset(&line_info,0,sizeof(line_info));
	line_info.line_offset=24;

	ioctl_result=ioctl(fd,GPIO_GET_LINEINFO_IOCTL,&line_info);
	if (ioctl_result<0 ) {
		fprintf(stderr,"Error ioctl %s\n",strerror(errno));
		close(fd);
		return -1;
        }

        printf("Offset %d, flags %x, name %s, consumer %s\n",
                line_info.line_offset,
                line_info.flags,
                line_info.name,
                line_info.consumer);


	*fd24=req24.fd;
	*fd25=req25.fd;


	return fd;
}

/* close a gpio */
static int close_gpio(int fd) {

	return close(fd);
}

/* set a GPIO low */
static int gpio_set_low(int fd) {

	int ioctl_result;
        struct gpiohandle_data data;

	data.values[0]=0;
	ioctl_result=ioctl(fd,GPIOHANDLE_SET_LINE_VALUES_IOCTL,&data);
	if (ioctl_result<0) {
		fprintf(stderr,"Error setting value %s\n",strerror(errno));
		return -1;
	}

	return 0;
}

/* set a gpio high */
static int gpio_set_high(int fd) {

	int ioctl_result;
        struct gpiohandle_data data;

	data.values[0]=1;
	ioctl_result=ioctl(fd,GPIOHANDLE_SET_LINE_VALUES_IOCTL,&data);
	if (ioctl_result<0) {
		fprintf(stderr,"Error setting value %s\n",strerror(errno));
		return -1;
	}

	return 0;
}

/* read a gpio value */
static int gpio_read(int fd) {
	struct gpiohandle_data data;
	int ioctl_result;

	memset(&data, 0, sizeof(data));
	ioctl_result = ioctl(fd, GPIOHANDLE_GET_LINE_VALUES_IOCTL, &data);
	if (ioctl_result<0) {
		fprintf(stderr,"Error reading!\n");
		return -1;
	}

	return data.values[0];
}


/* pthread thread */
/* wait roughly 100ms, then set GPIO24 high */
/*	also log the time it set things high */
static void *write_thread(void *argument) {

	int i;
	int *fd;

	fd=(int *)argument;


	for(i=0;i<max_count;i++) {
		gpio_set_low(*fd);
		//printf("Set GPIO24 to 0\n");
		usleep(100000);
		clock_gettime(CLOCK_REALTIME, &set_hi[i]);
		gpio_set_high(*fd);
		usleep(100000);
	}

	return NULL;
}

/* pthread thread */
/* watch GPIO25 and log when we notice it going high */
static void *read_thread(void *argument) {

	int count=0;
	int *fd;

	fd=(int *)argument;

	/* wait until is 0 */
	while(gpio_read(*fd)!=0) ;

	while(1) {

		while(gpio_read(*fd)==0) ;
		clock_gettime(CLOCK_REALTIME, &got_hi[count]);

//		printf("Went high %d!\n",count);
		count++;
		if (count>=max_count) break;

		while(gpio_read(*fd)==1) ;
//		printf("Went low %d!\n",count);
		if (count%10==0) printf("Read %d\n",count);


	}

	return NULL;
}



int main(int argc, char **argv) {

	int fd,fd24,fd25;
	long int diff;
	int result;
	pthread_t threads[2];
	int i;
	long int min=INT_MAX,max=0;
	double average=0;

	if (argc>1) {
		max_count=atoi(argv[1]);
		printf("Setting max_count to %d\n",max_count);
	}
	else {
		printf("max_count is %d\n",max_count);
	}

	/* allocate the result arrays */
	set_hi=calloc(max_count,sizeof(struct timespec));
	got_hi=calloc(max_count,sizeof(struct timespec));
	if ((set_hi==NULL)||(got_hi==NULL)) {
		fprintf(stderr,"Error allocating memory...\n");
		return -1;
	}

	/* init gpios */
	fd=init_gpio(&fd24,&fd25);

	/* start the writing thread */
	result = pthread_create(
                &threads[0],
                NULL,
                write_thread,
                (void *)&fd24);

	/* start the reading thread */
	result = pthread_create(
                &threads[1],
                NULL,
                read_thread,
                (void *)&fd25);

	(void)result;

	/* wait for both threads to finish */
	pthread_join(threads[0],NULL);
	pthread_join(threads[1],NULL);

	/* close open gpios */
	close_gpio(fd24);
	close_gpio(fd25);
	close_gpio(fd);

	/* calculate the average, min, and max */
	for(i=0;i<max_count;i++) {
		diff=calc_diff(set_hi[i],got_hi[i]);
		if (diff<min) min=diff;
		if (diff>max) max=diff;
		average+=diff;
	}

	printf("Out of %d tries:\n",max_count);
	printf("\tmin: %ld ns\n",min);
	printf("\tmax: %ld ns\n",max);
	printf("\taverage: %.0lf ns\n",average/max_count);

	return 0;
}
