#include <stddef.h>
#include <stdint.h>

#include "drivers/bcm2835/bcm2835_io.h"
#include "drivers/bcm2835/bcm2835_periph.h"
#include "drivers/keyboard/ps2-keyboard.h"
#include "drivers/timer/timer.h"
#include "drivers/serial/serial.h"

#include "lib/printk.h"
#include "lib/mmio.h"
#include "lib/smp.h"

#include "time/time.h"
#include "processes/scheduler.h"
#include "processes/exit.h"
#include "processes/process.h"

#include "interrupts/interrupts.h"
#include "interrupts/gic-400.h"

#include "boot/hardware_detect.h"

#define MAX_IRQ	64

int platform_irq_enable(void) {

	/* Enable Pi4 gic400 Interrupt Controller */

	if (hardware_type==RPI_MODEL_4B) {
		printk("Enabling GIC-400 on Pi4B\n");
		gic400_init((void *)0xFF840000UL);
	}

	return 0;

}

int irq_enable(int which_one) {

	uint32_t address_offset,bit,old;

        if (which_one>MAX_IRQ) {
                printk("IRQ%d too big\n",which_one);
                return -1;
        }

        bit=1<<(which_one&0x1f);
        address_offset=(which_one/32)*4;

        old=bcm2835_read(IRQ_ENABLE_IRQ1+address_offset);
        bcm2835_write(IRQ_ENABLE_IRQ1+address_offset,old|bit);

        return 0;
}


int irq_disable(int which_one) {

	uint32_t address_offset,bit,old;

        if (which_one>MAX_IRQ) {
                printk("IRQ%d too big\n",which_one);
                return -1;
        }

        bit=1<<(which_one&0x1f);
        address_offset=(which_one/32)*4;

        old=bcm2835_read(IRQ_DISABLE_IRQ1+address_offset);
        bcm2835_write(IRQ_DISABLE_IRQ1+address_offset,old|bit);

        return 0;

}

static void user_reg_dump(uint32_t *regs) {

	int i;

	printk("Process: %d (text %p+%x, stack %p+%x)\n",
		current_proc[get_cpu()]->pid,
		current_proc[get_cpu()]->text,
		current_proc[get_cpu()]->textsize,
		current_proc[get_cpu()]->stack,
		current_proc[get_cpu()]->stacksize);

	for(i=0;i<8;i++) {
		printk("r%02d: %08x\tr%02d: %08x",i,regs[i],i+8,regs[i+8]);
		printk("\n");
	}


}

void interrupt_handler_c(void) {

	uint32_t basic_pending,pending0,pending1,pending2;
	uint32_t handled=0,was_timer=0;

	/* Ideally we'd have a proper irq registration mechanism for this */
	if (hardware_type==RPI_MODEL_4B) {

		/* Note, based on IRQ_PENDING2 you can tell */
		/* 	if there's an interrupt in PENDING0/PENDING1 */
		pending0=bcm2835_read(IRQ0_PENDING0);
		pending1=bcm2835_read(IRQ0_PENDING1);
		pending2=bcm2835_read(IRQ0_PENDING2);

		// Check if UART (irq57)
		if (pending1 & IRQ_PENDING1_IRQ57) {
			handled++;
			serial_interrupt_handler();
		}

		// check if it's a timer interrupt
		if (pending2 & IRQ_PENDING2_TIMER_IRQ) {
			was_timer++;
			handled++;
			/* we handle this later */
		}

		if (!handled) {
			printk("Unknown interrupt happened %x %x %x!\n",
					pending0,pending1,pending2);
			/* clear pending */
			mmio_write(GICD_ICPENDR0,0xffffffffUL);
			return;
		}

	}

	else {

		/**************************************/
		/* First check basic_pending register */
		/**************************************/
		basic_pending=bcm2835_read(IRQ_BASIC_PENDING);

		if (basic_pending) {
			/* check serial port */
			if (basic_pending & IRQ_BASIC_PENDING_IRQ57) {
				handled++;
				serial_interrupt_handler();
			}

			/* check if it's a timer interrupt */
			if (basic_pending & IRQ_BASIC_PENDING_TIMER) {
				handled++;
				was_timer++;
			}

			if (!handled) {
				printk("Unknown basic_pending interrupt %x!\n",
					basic_pending);
				return;
			}
		}
		/*************************************/
		/* Next check pending1 register      */
		/*************************************/
		pending1=bcm2835_read(IRQ_PENDING1);
		if (pending1) {
			if (!handled) {
				printk("Unknown pending1 interrupt %x\n",
					pending1);
			}
		}

		/*************************************/
		/* Next check pending2 register      */
		/*************************************/
		pending2=bcm2835_read(IRQ_PENDING2);

		if (pending2) {
			/* Check if GPIO23 (ps2 keyboard) (irq49) */
#if 0
			if (pending2 & IRQ_PENDING2_IRQ49) {
				handled++;
				ps2_interrupt_handler();
			}
#endif
			if (!handled) {
				printk("Unknown pending2 interrupt %x\n",
					pending2);
			}

                }

	}

	if (!handled) {
		printk("Unknown interrupt!\n");
	}

	/* FIXME: do this properly */
	if (hardware_type == RPI_MODEL_4B) {
		mmio_write(GICD_ICPENDR0,0xffffffffUL);        // clear pending
	}

	/* check if it was a timer interrupt */
	if (was_timer) {
//		printk("About to handle timer interrupt\n");
		timer_interrupt_handler();
//		printk("Returned from timer interrupt\n");
		if (scheduling_enabled) schedule();
	}

//      printk("Exiting interrupt_c\n");

	return;
}

void __attribute__((interrupt("FIQ"))) fiq_handler(void) {
	printk("UNHANDLED FIQ\n");
}

/* 1415 */
//void __attribute__((interrupt("ABORT"))) data_abort_handler(void) {

void data_abort_handler_c(uint32_t *register_array) {

	uint32_t dfsr,dfar,fs;
//	register long lr asm ("lr");
	uint32_t lr;
	uint32_t *code;

	/* In theory we want LR-8 here, but */
	/* entry code already subtracted 4 */
	lr=register_array[15];
	code=(uint32_t *)(lr-4);

	printk("MEMORY ABORT at PC=%x (%x)\n",lr-4,*code);

	/* Read DFSR reg (see B4.1.52) */

	asm volatile("mrc p15, 0, %0, c5, c0, 0" : "=r" (dfsr) : : "cc");

	/* Read DFAR reg (see B4.1.52) */
	/* Data Fault Address Register */
	asm volatile("mrc p15, 0, %0, c6, c0, 0" : "=r" (dfar) : : "cc");

	fs=dfsr&0xf;

	if (fs==1) printk("\tAlignment fault\n");
	if (fs==2) printk("\tDebug event\n");
	if ((fs&0xd)==0xd) {
	        printk("\tPermission fault %s %x\n",
                        (dfsr&(1<<11))?"writing":"reading",dfar);
        }

	user_reg_dump(register_array);

	exit(-1);

}

/* 1637 */
void __attribute__((interrupt("ABORT"))) prefetch_abort_handler(void) {
	uint32_t ifsr,ifar,fs;
	register long lr asm ("lr");

	printk("PREFETCH ABORT at PC=%x\n",lr-4);

	/* Read IFSR reg (see B4.1.96) */
	asm volatile("mrc p15, 0, %0, c5, c0, 1" : "=r" (ifsr) : : "cc");

	/* Read IFAR reg (see B4.1.96) */
	asm volatile("mrc p15, 0, %0, c6, c0, 2" : "=r" (ifar) : : "cc");

	fs=ifsr&0xf;

	if (fs==2) printk("\tDebug event\n");
	if ((fs&0xd)==0xd) printk("\tPermission fault accessing %x\n",ifar);

	// user_reg_dump()

	exit(-1);

}


void __attribute__((interrupt("UNDEF"))) undef_handler(void) {

	register long lr asm ("lr");

	long *ptr=(long *)lr;

	/* point back 4 bytes */
	ptr--;

	printk("UNDEFINED INSTRUCTION, PC=%x, insn=%x\n",lr-4,*ptr);

}
